<?php
namespace Fifa\ConnectServiceBus\Sdk\Authentication;

use Fifa\ConnectServiceBus\Sdk\Authentication\Model\AuthenticationInformation;
use Fifa\ConnectServiceBus\Sdk\Authentication\Model\Token;
use Fifa\ConnectServiceBus\Sdk\Utils\Cache;

/**
 * Class TokenCache
 * @package Fifa\ConnectServiceBus\Sdk\Authentication
 */
class TokenCache extends Cache implements TokenCacheInterface
{
    /**
     * @var Token
     */
    private $token;

    /**
     * @var AuthenticationInformation
     */
    private $authenticationInformation;

    /**
     * Constructor
     *
     * @param AuthenticationInformation $authenticationInformation
     * @param CacheStorageInterface $cacheStorage
     * @param int $validLimit
     */
    public function __construct(AuthenticationInformation $authenticationInformation, CacheStorageInterface $cacheStorage, $validLimit = 300)
    {
        parent::__construct($cacheStorage);
        $this->setValidLimit($validLimit);
        $this->authenticationInformation = $authenticationInformation;
    }

    /**
     * Checks whether stored cache is still valid
     *
     * @return bool
     */
    public function isValid()
    {
        $this->loadCache();
        $diff = $this->token->getExpiresOn() - time();

        return $diff >= $this->getValidLimit();
    }

    /**
     * Returns token object that is stored in cache
     *
     * @return Token
     */
    public function getTokenFromCache()
    {
        if (is_null($this->token)) {
            $this->loadCache();
        }

        return $this->token;
    }

    /**
     * Store given token object into cache
     *
     * @param Token $token
     * @return TokenCacheInterface
     */
    public function storeCache(Token $token)
    {
        $this->putIntoCache($token, $this->authenticationInformation);
        $this->token = $token;

        return $this;
    }

    /**
     * Loads cache from file into memory
     */
    private function loadCache()
    {
        $this->token = $this->getFromCache($this->authenticationInformation);
        if (is_null($this->token) || !$this->token) {
            $this->token = new Token(new \stdClass());
        }
    }
}