<?php
namespace Fifa\ConnectServiceBus\Sdk\Authentication;

use Fifa\ConnectServiceBus\Sdk\Api\ApiException;
use Fifa\ConnectServiceBus\Sdk\Authentication\Model\AuthenticationInformation;
use Fifa\ConnectServiceBus\Sdk\Authentication\Model\Token;
use Fifa\ConnectServiceBus\Sdk\Exception\AuthenticationException;

/**
 * Class TokenProvider
 * @package Fifa\ConnectServiceBus\Sdk\Authentication
 */
class TokenProvider implements TokenProviderInterface
{
    /**
     * @var AuthenticationInformation
     */
    private $authenticationInformation;

    /**
     * @var TokenCacheInterface
     */
    private $tokenCache;

    /**
     * Constructor
     * @param AuthenticationInformation $authenticationInformation
     * @param CacheStorageInterface $cacheStorage
     */
    public function __construct(AuthenticationInformation $authenticationInformation, CacheStorageInterface $cacheStorage)
    {
        $this->authenticationInformation = $authenticationInformation;
        $this->tokenCache = new TokenCache($authenticationInformation, $cacheStorage);
    }

    /**
     * Returns the token for API authorization
     *
     * @return Token
     * @throws \Fifa\ConnectServiceBus\Sdk\Exception\AuthenticationException
     */
    public function getToken()
    {
        if ($this->tokenCache->isValid()) {
            $token = $this->tokenCache->getTokenFromCache();
        }
        else {
            $token = $this->getAuthenticationHeader();
            $this->tokenCache->storeCache($token);
        }

        return $token;
    }

    /**
     * Post the token generated from the symetric key and other information to STS URL and construct the authentication header
     *
     * @return Token
     * @throws \Fifa\ConnectServiceBus\Sdk\Exception\AuthenticationException
     */
    private function getAuthenticationHeader()
    {
        $clientSecret = urlencode($this->authenticationInformation->getSecretKey());
        // Information about the resource we need access for which in this case is graph.
        $graphId = $this->authenticationInformation->getResourceUri();
        $graphPrincipalId = urlencode($graphId);
        // Information about the app
        $clientPrincipalId = urlencode($this->authenticationInformation->getClientId());

        // Construct the body for the STS request
        $authenticationRequestBody =
            'grant_type=client_credentials&client_secret=' . $clientSecret . '&resource=' . $graphPrincipalId . '&client_id=' . $clientPrincipalId;

        //Using curl to post the information to STS and get back the authentication response
        $ch = curl_init();
        // set url
        $stsUrl = 'https://login.windows.net/' . $this->authenticationInformation->getTenant(). '/oauth2/token?api-version=1.0';
        curl_setopt($ch, CURLOPT_URL, $stsUrl);
        // Get the response back as a string
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        // Mark as Post request
        curl_setopt($ch, CURLOPT_POST, 1);
        // Set the parameters for the request
        curl_setopt($ch, CURLOPT_POSTFIELDS,  $authenticationRequestBody);

        // By default, HTTPS does not work with curl.
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        // read the output from the post request
        $output = curl_exec($ch);
        // close curl resource to free up system resources
        curl_close($ch);
        // decode the response from sts using json decoder
        $tokenOutput = json_decode($output);

        if (!isset($tokenOutput->access_token)) {
            $description = isset($tokenOutput->error_description) ? $tokenOutput->error_description : '';
            $apiException = new ApiException(
                'Failed getting the access token. ' . $description
            );
            throw new AuthenticationException($apiException, $description);
        }

        return new Token($tokenOutput);
    }
}