<?php
namespace Fifa\ConnectServiceBus\Sdk\Encryption;

use Fifa\ConnectServiceBus\Sdk\Authentication\CacheStorageInterface;
use Fifa\ConnectServiceBus\Sdk\Encryption\Model\CacheItem;
use Fifa\ConnectServiceBus\Sdk\Encryption\Model\PublicCertificate;
use Fifa\ConnectServiceBus\Sdk\Utils\Cache;

/**
 * Class CertificateCache
 * @package Fifa\ConnectServiceBus\Sdk\Encryption
 */
class CertificateCache extends Cache implements CertificateCacheInterface
{
    /**
     * @var CacheItem[]
     */
    private $cacheItems;

    /**
     * Constructor
     *
     * @param CacheStorageInterface $cacheStorage
     * @param int $validLimit - for how long cache should be valid until it's created, in seconds (default 86400 = 1 day)
     */
    public function __construct(CacheStorageInterface $cacheStorage, $validLimit = 86400)
    {
        parent::__construct($cacheStorage);
        $this->setValidLimit($validLimit);
    }

    /**
     * Checks whether stored cache is still valid
     *
     * @param string $queueIdentifier
     * @return bool
     */
    private function isValid($queueIdentifier)
    {
        $this->loadCache($queueIdentifier);
        if (!isset($this->cacheItems[$queueIdentifier]) || !$this->cacheItems[$queueIdentifier]) {
            return false;
        }

        $diff = $this->cacheItems[$queueIdentifier]->getExpiresOn() - time();

        return $diff >= 0;
    }

    /**
     * Returns object that is stored in cache
     *
     * @param string $queueIdentifier
     * @return PublicCertificate|null
     */
    public function getCertificateFromCache($queueIdentifier)
    {
        if (!is_string($queueIdentifier)) {
            throw new \InvalidArgumentException('Organisation ID needs to be a string.');
        }

        if (!$this->isValid($queueIdentifier)) {
            return null;
        }

        if (!isset($this->cacheItems[$queueIdentifier]) || is_null($this->cacheItems[$queueIdentifier])) {
            $this->loadCache($queueIdentifier);
        }

        return $this->cacheItems[$queueIdentifier]->getData();
    }

    /**
     * Store given certificate into cache
     *
     * @param PublicCertificate $certificate
     */
    public function storeCache(PublicCertificate $certificate)
    {
        $cacheItem = new CacheItem($certificate, time() + $this->getValidLimit());
        $this->putIntoCache($cacheItem, $this->getCacheFilenameKey($certificate->getQueueIdentifier()));
        $this->cacheItems[$certificate->getQueueIdentifier()] = $cacheItem;
    }

    /**
     * Loads cache from file into memory
     *
     * @param string $queueIdentifier
     */
    private function loadCache($queueIdentifier)
    {
        $this->cacheItems[$queueIdentifier] = $this->getFromCache($this->getCacheFilenameKey($queueIdentifier));
        if (!isset($this->cacheItems[$queueIdentifier]) || is_null($this->cacheItems[$queueIdentifier])) {
            $this->cacheItems[$queueIdentifier] = new CacheItem(new PublicCertificate('', ''), 0);
        }
    }

    /**
     * Returns the string that will be used as a token to create a cache file.
     *
     * @param string $queueIdentifier
     * @return string
     */
    private function getCacheFilenameKey($queueIdentifier)
    {
        $publicCertificateClassFilePath = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'Model' . DIRECTORY_SEPARATOR . 'PublicCertificate.php';
        if (!file_exists($publicCertificateClassFilePath)) {
            throw new \InvalidArgumentException('PublicCertificate.php file path is incorrect.');
        }

        $publicCertificateContent = file_get_contents($publicCertificateClassFilePath);
        return $queueIdentifier . $publicCertificateContent;
    }
}