<?php
namespace Fifa\ConnectServiceBus\Sdk\Encryption\Encrypt;

use Fifa\ConnectServiceBus\Sdk\Api\ApiException;
use Fifa\ConnectServiceBus\Sdk\Exception\DataNotFoundException;
use Fifa\ConnectServiceBus\Sdk\Exception\PublicCertificateNotFoundException;
use Fifa\ConnectServiceBus\Sdk\FifaConnectServiceBusCertificateClientInterface;
use Fifa\ConnectServiceBus\Sdk\FifaConnectServiceBusClientInterface;

/**
 * Class PublicCertificateSource
 * @package Fifa\ConnectServiceBus\Sdk\Encryption\Encrypt
 */
class PublicCertificateSource implements PublicCertificateSourceInterface
{
    /**
     * @var FifaConnectServiceBusClientInterface
     */
    private $certificateClient;

    /**
     * @var PublicKeyStorageInterface
     */
    private $publicKeyCache;

    /**
     * Constructor
     *
     * @param PublicKeyStorageInterface $publicKeyCache
     * @param FifaConnectServiceBusCertificateClientInterface $client
     */
    public function __construct(PublicKeyStorageInterface $publicKeyCache, FifaConnectServiceBusCertificateClientInterface $client)
    {
        $this->certificateClient = $client;
        $this->publicKeyCache = $publicKeyCache;
    }

    /**
     * Gets public key
     *
     * @param string $queueIdentifier
     * @return string
     * @throws PublicCertificateNotFoundException
     */
    public function get($queueIdentifier)
    {
        $key = $this->publicKeyCache->get($queueIdentifier);

        if (is_null($key)) {

            try {
                $key = $this->certificateClient->downloadCertificate($queueIdentifier);
            }
            catch (DataNotFoundException $e) {
                $message = 'Organisation ' . $queueIdentifier . ' have not uploaded their public key.';
                $apiException = new ApiException($message, $e->getCode(), $e->getResponseHeaders(), $e->getResponseBody());
                throw new PublicCertificateNotFoundException($apiException);
            }

            $this->publicKeyCache->insert($queueIdentifier, $key);
        }

        return $key;
    }
}