<?php
namespace Fifa\ConnectServiceBus\Sdk;

use Fifa\ConnectServiceBus\Sdk\Api\ApiException;
use Fifa\ConnectServiceBus\Sdk\Api\Resource\CertificateApi;
use Fifa\ConnectServiceBus\Sdk\Authentication\Model\ClientCredentials;
use Fifa\ConnectServiceBus\Sdk\Environment\EnvironmentInterface;
use Fifa\ConnectServiceBus\Sdk\Exception\AuthenticationException;
use Fifa\ConnectServiceBus\Sdk\Exception\DataNotFoundException;
use Fifa\ConnectServiceBus\Sdk\Exception\FifaConnectServiceBusSdkException;
use Fifa\ConnectServiceBus\Sdk\Exception\InvalidClientDataException;
use Fifa\ConnectServiceBus\Sdk\Exception\UnauthorizedException;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\ApiExceptionHandler;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\Handler\DataNotFoundHandler;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\Handler\UnauthorizedRequestHandler;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

/**
 * Class FifaConnectServiceBusCertificateClient
 * @package Fifa\ConnectServiceBus\Sdk
 */
class FifaConnectServiceBusCertificateClient implements FifaConnectServiceBusCertificateClientInterface
{
    /**
     * @var CertificateApi
     */
    private $certificateApi;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Constructor
     *
     * @param EnvironmentInterface $environment
     * @param ClientCredentials $clientCredentials
     * @param LoggerInterface|null $logger
     */
    public function __construct(EnvironmentInterface $environment, ClientCredentials $clientCredentials, LoggerInterface $logger = null)
    {
        $this->logger = $logger ? $logger : new NullLogger();
        $this->certificateApi = $environment->getCertificateApi($clientCredentials);
    }

    /**
     * Upload a public certificate to central infrastructure.
     *
     * @param string $rawData - string containing certificate data
     *
     * @throws FifaConnectServiceBusSdkException
     * @throws InvalidClientDataException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function uploadCertificate($rawData)
    {
        if (is_null($rawData) || strlen($rawData) === 0) {
            throw new \InvalidArgumentException('Certificate file path is incorrect');
        }

        try {
            $this->logger->debug('Uploading certificate');

            $response = $this->certificateApi->certificateUploadByRawdataWithHttpInfo(
                json_encode(base64_encode($rawData))
            );

            $this->logger->debug('Uploading certificate has completed successfully', array('response' => $response));
        }
        catch (ApiException $e) {
            $this->logger->error('Unexpected error when trying to upload a certificate', array('exception' => $e));
            throw ApiExceptionHandler::createException($e);
        }
    }

    /**
     * Gets a byte array that contains data from an X.509 certificate.
     *
     * @param string $queueIdentifier - Organisation FIFA Unique Identifier
     * @return string
     *
     * @throws FifaConnectServiceBusSdkException
     * @throws InvalidClientDataException
     * @throws DataNotFoundException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function downloadCertificate($queueIdentifier)
    {
        try {
            $this->logger->debug('Downloading certificate');
            $response = $this->certificateApi->certificateDownloadByQueueidentifier($queueIdentifier);

            if (!is_string($response[0]) || strlen($response[0]) < 2) {

                $message = 'Certificate is not a valid string.';
                $this->logger->error($message);
                throw new FifaConnectServiceBusSdkException(new ApiException($message), $message);
            }

            $this->logger->debug('Downloading certificate has completed successfully', array('response' => $response));
            return base64_decode($response[0]);
        }
        catch (ApiException $e) {
            $this->logger->error('Unexpected error when trying to download a certificate', array('exception' => $e));
            throw ApiExceptionHandler::createException($e, new DataNotFoundHandler(), new UnauthorizedRequestHandler());
        }
    }
}