<?php
namespace Fifa\ConnectServiceBus\Sdk;

use Fifa\ConnectServiceBus\Sdk\Api\ApiException;
use Fifa\ConnectServiceBus\Sdk\Authentication\Model\ClientCredentials;
use Fifa\ConnectServiceBus\Sdk\Encryption\CertificateBasedCryptographyAlgorithm;
use Fifa\ConnectServiceBus\Sdk\Encryption\CertificateCache;
use Fifa\ConnectServiceBus\Sdk\Encryption\Decrypt\MessageDecryptionServiceInterface;
use Fifa\ConnectServiceBus\Sdk\Encryption\Decrypt\PrivateCertificateStorageInterface;
use Fifa\ConnectServiceBus\Sdk\Encryption\Encrypt\MessageEncryptionServiceInterface;
use Fifa\ConnectServiceBus\Sdk\Encryption\Encrypt\PublicCertificateProvider;
use Fifa\ConnectServiceBus\Sdk\Encryption\Encrypt\PublicCertificateSource;
use Fifa\ConnectServiceBus\Sdk\Encryption\Encrypt\PublicKeyMemoryStorage;
use Fifa\ConnectServiceBus\Sdk\Encryption\MessageCryptographyService;
use Fifa\ConnectServiceBus\Sdk\Encryption\RSACryptographyService;
use Fifa\ConnectServiceBus\Sdk\Environment\EnvironmentInterface;
use Fifa\ConnectServiceBus\Sdk\Exception\AuthenticationException;
use Fifa\ConnectServiceBus\Sdk\Exception\FifaConnectServiceBusSdkException;
use Fifa\ConnectServiceBus\Sdk\Exception\InvalidClientDataException;
use Fifa\ConnectServiceBus\Sdk\Exception\MessageNotFoundException;
use Fifa\ConnectServiceBus\Sdk\Exception\NoMessagesAvailableException;
use Fifa\ConnectServiceBus\Sdk\Exception\QueueNotFoundException;
use Fifa\ConnectServiceBus\Sdk\Exception\UnauthorizedException;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\ApiExceptionHandler;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\Handler\InvalidClientDataHandler;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\Handler\MessageNotFoundHandler;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\Handler\NoMessagesAvailableHandler;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\Handler\QueueAlreadyExistsHandler;
use Fifa\ConnectServiceBus\Sdk\ExceptionHandler\Handler\QueueNotFoundHandler;
use Fifa\ConnectServiceBus\Sdk\Message\Message;
use Fifa\ConnectServiceBus\Sdk\Api\Resource\MessageApi;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

/**
 * Class FifaConnectServiceBusClient
 * @package Fifa\ConnectServiceBus\Sdk
 */
class FifaConnectServiceBusClient implements FifaConnectServiceBusClientInterface
{
    /**
     * @var MessageApi
     */
    private $messageApi;

    /**
     * @var bool
     */
    private $useEncryption = true;

    /**
     * @var MessageEncryptionServiceInterface
     */
    private $encryptionService;

    /**
     * @var MessageDecryptionServiceInterface
     */
    private $decryptionService;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Constructor
     *
     * @param MessageApi $messageApi
     * @param MessageEncryptionServiceInterface $encryptionService
     * @param MessageDecryptionServiceInterface $decryptionService
     * @param LoggerInterface $logger
     */
    public function __construct(MessageApi $messageApi, MessageEncryptionServiceInterface $encryptionService,
                                MessageDecryptionServiceInterface $decryptionService, LoggerInterface $logger)
    {
        $this->messageApi = $messageApi;
        $this->decryptionService = $decryptionService;
        $this->logger = $logger;
        $this->encryptionService = $encryptionService;
    }

    /**
     * Creates an instance of FifaConnectServiceBusClient
     *
     * @param EnvironmentInterface $environment
     * @param ClientCredentials $clientCredentials
     * @param PrivateCertificateStorageInterface $privateCertificateStorage
     * @param LoggerInterface|null $logger
     * @return FifaConnectServiceBusClient
     */
    public static function create(EnvironmentInterface $environment, ClientCredentials $clientCredentials,
                                  PrivateCertificateStorageInterface $privateCertificateStorage, LoggerInterface $logger = null)
    {
        if (is_null($logger)) {
            $logger = new NullLogger();
        }

        $algorithms = array(
            new CertificateBasedCryptographyAlgorithm(new RSACryptographyService(
                new PublicCertificateProvider(
                    new PublicCertificateSource(
                        new PublicKeyMemoryStorage(), new FifaConnectServiceBusCertificateClient($environment, $clientCredentials, $logger)
                    ),
                    new CertificateCache($environment->getCacheStorage())
                ),
                $privateCertificateStorage,
                $logger
            ), $logger)
        );

        $messageService = new MessageCryptographyService($algorithms);
        $messageApi = $environment->getMessageApi($clientCredentials);

        return new FifaConnectServiceBusClient($messageApi, $messageService, $messageService, $logger);
    }

    /**
     * Sends a message to a specified recipient via Message Bus.
     *
     * @param string $recipient
     * @param Message $message
     * @return null
     *
     * @throws InvalidClientDataException
     * @throws QueueNotFoundException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function send($recipient, Message $message)
    {
        $this->logger->debug(sprintf('Sending message to %s', $recipient), array('message' => $message, 'recipient' => $recipient));
        try {

            $content = $message->getContent();
            $properties = $message->getProperties();

            if ($this->useEncryption) {

                $this->logger->debug('Encrypting message to be sent to: ' . $recipient);
                $encryptionResult = $this->encryptionService->encrypt($content, $recipient);

                $content = $encryptionResult->getData();
                $properties = array_merge($properties, $encryptionResult->getProperties());
            }
            else {
                $this->logger->debug('Encryption is turned off - original message is going to be sent.');
            }

            list($response, $statusCode, $httpHeaders) = $this->messageApi->messageSendByRecipientAndBodyAndActionAndPropertiesWithHttpInfo(
                $recipient,
                json_encode(base64_encode($content)),
                $message->getAction(),
                json_encode($properties)
            );

            if ($statusCode != self::HTTP_CODE_CREATED) {
                throw new ApiException('Unexpected result', $statusCode, $httpHeaders, $response);
            }

            $this->logResponse($response, $statusCode, $httpHeaders);
        }
        catch (ApiException $e) {
            $this->logger->error('Sending message error.', array('exception' => $e));

            throw ApiExceptionHandler::createException($e, new QueueNotFoundHandler());
        }
    }

    /**
     * Gets a message from the Message Bus. Message is deleted from the queue.
     *
     * @param int $timeout
     * @return Message
     *
     * @throws NoMessagesAvailableException
     * @throws InvalidClientDataException
     * @throws QueueNotFoundException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function receive($timeout = 60)
    {
        $this->logger->debug(sprintf('Trying to receive message'), array('timeout' => $timeout));
        try {
            list($response, $statusCode, $httpHeaders) = $this->messageApi->messageReceiveByRequestTimeoutWithHttpInfo($timeout);

            $this->logResponse($response, $statusCode, $httpHeaders);
            $message = $this->createMessage($response, $statusCode, $httpHeaders);

            $properties = $message->getProperties();
            $isEncryptionUsed = isset($properties[MessageCryptographyService::ENCRYPTION_VERSION_KEY]);

            if ($isEncryptionUsed) {

                $this->logger->debug('Message received from "' . $message->getSender() . '" appears to be encrypted. It will be now decrypted.');
                $message->setContent($this->decryptionService->decrypt($message->getContent(), $properties));
            }
            else {
                $this->logger->debug('Message received from "' . $message->getSender() . '" does not appear to be encrypted. Original content is going to be returned.');
            }

            return $message;
        }
        catch (ApiException $e) {
            $this->logger->error('Receiving message error.', array('exception' => $e));

            throw ApiExceptionHandler::createException($e, new NoMessagesAvailableHandler(), new QueueNotFoundHandler());
        }
    }

    /**
     * Gets and locks a message without deleting it from the queue. If message is not deleted within configured time it's automatically unlocked and visible for other recipients.
     *
     * @param int $timeout
     * @return Message
     *
     * @throws NoMessagesAvailableException
     * @throws QueueNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function peekLock($timeout = 60)
    {
        $this->logger->debug(sprintf('Creating peek lock'), array('timeout' => $timeout));
        try {
            list($response, $statusCode, $httpHeaders) = $this->messageApi->messagePeekLockByRequestTimeoutWithHttpInfo($timeout);

            $this->logResponse($response, $statusCode, $httpHeaders);
            $message = $this->createMessage($response, $statusCode, $httpHeaders, self::HTTP_CODE_CREATED);

            $properties = $message->getProperties();
            $isEncryptionUsed = isset($properties[MessageCryptographyService::ENCRYPTION_VERSION_KEY]);

            if ($isEncryptionUsed) {

                $this->logger->debug('Message received from "' . $message->getSender() . '" appears to be encrypted. It will be now decrypted.');
                $message->setContent($this->decryptionService->decrypt($message->getContent(), $properties));
            }
            else {
                $this->logger->debug('Message received from "' . $message->getSender() . '" does not appear to be encrypted. Original content is going to be returned.');
            }

            return $message;
        }
        catch (ApiException $e) {
            $this->logger->error('Creating peek lock error.', array('exception' => $e));

            throw ApiExceptionHandler::createException($e, new QueueNotFoundHandler(), new NoMessagesAvailableHandler());
        }
    }

    /**
     * Deletes a previously locked message (see peekLock method).
     *
     * @param string $messageId
     * @param string $lockToken
     * @return null
     *
     * @throws MessageNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function delete($messageId, $lockToken)
    {
        $this->logger->debug(sprintf('Deleting message id "%s" (lock token "%s")', $messageId, $lockToken), array(
            'messageId' => $messageId,
            'lockToken' => $lockToken,
        ));
        try {
            list($response, $statusCode, $httpHeaders) = $this->messageApi->messageDeleteByRequestMessageidAndRequestLocktokenWithHttpInfo($messageId, $lockToken);

            if ($statusCode != self::HTTP_CODE_OK) {
                throw new ApiException('Unexpected result', $statusCode, $httpHeaders, $response);
            }

            $this->logResponse($response, $statusCode, $httpHeaders);
        }
        catch (ApiException $e) {
            $this->logger->error('Deleting message error.', array('exception' => $e));

            throw ApiExceptionHandler::createException($e, new MessageNotFoundHandler(), new InvalidClientDataHandler(410));
        }
    }

    /**
     * Unlocks a previously locked message. The message is returned to the queue and can be fetched using peekLock or receive method.
     *
     * @param string $messageId
     * @param string $lockToken
     * @return null
     *
     * @throws MessageNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function unlock($messageId, $lockToken)
    {
        $this->logger->debug(sprintf('Unlocking message id "%s" (lock token "%s")', $messageId, $lockToken), array(
            'messageId' => $messageId,
            'lockToken' => $lockToken,
        ));
        try {
            list($response, $statusCode, $httpHeaders) = $this->messageApi->messageUnlockByRequestMessageidAndRequestLocktokenWithHttpInfo($messageId, $lockToken);

            if ($statusCode != self::HTTP_CODE_OK) {
                throw new ApiException('Unexpected result.', $statusCode, $httpHeaders, $response);
            }
            $this->logResponse($response, $statusCode, $httpHeaders);
        }
        catch (ApiException $e) {
            $this->logger->error('Unlocking message error.', array('exception' => $e));

            throw ApiExceptionHandler::createException($e, new MessageNotFoundHandler(), new InvalidClientDataHandler(410));
        }
    }

    /**
     * Extends the lock time for the previously locked message (see peekLock method).
     *
     * @param string $messageId
     * @param string $lockToken
     * @return null
     *
     * @throws MessageNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function renewLock($messageId, $lockToken)
    {
        $this->logger->debug(sprintf('Renewing lock for message "%s" (lock token "%s")', $messageId, $lockToken), array(
            'messageId' => $messageId,
            'lockToken' => $lockToken,
        ));
        try {
            list($response, $statusCode, $httpHeaders) = $this->messageApi->messageRenewLockByRequestMessageidAndRequestLocktokenWithHttpInfo($messageId, $lockToken);

            if ($statusCode != self::HTTP_CODE_OK) {
                throw new ApiException('Unexpected result', $statusCode, $httpHeaders, $response);
            }
            $this->logResponse($response, $statusCode, $httpHeaders);
        }
        catch (ApiException $e) {
            $this->logger->error('Renewing lock error.', array('exception' => $e));

            throw ApiExceptionHandler::createException($e, new MessageNotFoundHandler(), new InvalidClientDataHandler(410));
        }
    }

    /**
     * Creates a topic with given name
     *
     * @param $topicName
     *
     * @throws FifaConnectServiceBusSdkException
     * @throws InvalidClientDataException
     * @throws MessageNotFoundException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function createTopic($topicName)
    {
        if (!is_null($topicName) && strlen($topicName) === 0) {
            throw new \InvalidArgumentException('Topic name needs to be a not empty string.');
        }

        $this->logger->debug('Creating a new topic with name ' . $topicName);
        try {

            $this->messageApi->messageCreateTopicByTopicnameAndShardnumber($topicName);
        }
        catch (ApiException $e) {

            $this->logger->error('Error when creating a new topic with name: ' . $topicName);
            throw ApiExceptionHandler::createException($e, new QueueAlreadyExistsHandler());
        }
    }

    /**
     * Deletes a topic with given name
     *
     * @param $topicName
     *
     * @throws FifaConnectServiceBusSdkException
     * @throws InvalidClientDataException
     * @throws MessageNotFoundException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function deleteTopic($topicName)
    {
        if (!is_null($topicName) && strlen($topicName) === 0) {
            throw new \InvalidArgumentException('Topic name needs to be a not empty string.');
        }

        $this->logger->debug('Deleting a topic with name ' . $topicName);
        try {

            $this->messageApi->messageDeleteTopicByTopicname($topicName);
        }
        catch (ApiException $e) {

            $this->logger->error('Error when deleting a topic with name: ' . $topicName);
            throw ApiExceptionHandler::createException($e, new QueueNotFoundHandler());
        }
    }

    /**
     * Returns an encryption flag
     *
     * @return boolean
     */
    public function isEncryptionInUse()
    {
        return $this->useEncryption;
    }

    /**
     * Sets encryption flag
     *
     * @param boolean $useEncryption
     * @return FifaConnectServiceBusClient
     */
    public function setUseEncryption($useEncryption)
    {
        $this->useEncryption = $useEncryption;
        return $this;
    }

    /**
     * @param array $response
     * @param int $statusCode
     * @param array $httpHeaders
     * @param int $expectedSuccessStatusCode
     * @return Message
     *
     * @throws ApiException
     * @throws NoMessagesAvailableException
     */
    private function createMessage($response, $statusCode, $httpHeaders, $expectedSuccessStatusCode = self::HTTP_CODE_OK)
    {
        if ($statusCode == self::HTTP_CODE_NO_CONTENT) {
            throw new ApiException('No messages available within the specified timeout period.', $statusCode, $httpHeaders, $response);
        }

        if ($statusCode == $expectedSuccessStatusCode && is_array($response) && count($response) == 1 && isset($response[0])) {
            return new Message(
                base64_decode((string) $response[0]),
                isset($httpHeaders['X-Action']) ? $httpHeaders['X-Action'] : null,
                isset($httpHeaders['X-Properties']) ? json_decode($httpHeaders['X-Properties'], true) : array(),
                isset($httpHeaders['BrokerProperties']) ? json_decode($httpHeaders['BrokerProperties'], true) : array()
            );
        }

        throw new ApiException('Unexpected result', $statusCode, $httpHeaders, $response);
    }

    /**
     * @param mixed $response
     * @param int $statusCode
     * @param array $httpHeaders
     * @param string $logMessage
     */
    private function logResponse($response, $statusCode, $httpHeaders, $logMessage = null)
    {
        $logMessage = is_string($logMessage) ? $logMessage : 'Received response from Connect Service Bus service.';

        $this->logger->debug($logMessage, array(
            'response'   => $response,
            'statusCode' => $statusCode,
            'httpHeader' => $httpHeaders
        ));
    }
};