<?php
namespace Fifa\ConnectServiceBus\Sdk;

use Fifa\ConnectServiceBus\Sdk\Exception\AuthenticationException;
use Fifa\ConnectServiceBus\Sdk\Exception\FifaConnectServiceBusSdkException;
use Fifa\ConnectServiceBus\Sdk\Exception\InvalidClientDataException;
use Fifa\ConnectServiceBus\Sdk\Exception\MessageNotFoundException;
use Fifa\ConnectServiceBus\Sdk\Exception\NoMessagesAvailableException;
use Fifa\ConnectServiceBus\Sdk\Exception\QueueNotFoundException;
use Fifa\ConnectServiceBus\Sdk\Exception\UnauthorizedException;
use Fifa\ConnectServiceBus\Sdk\Message\Message;

/**
 * Interface FifaConnectServiceBusClientInterface
 * @package Fifa\ConnectServiceBus\Sdk
 */
interface FifaConnectServiceBusClientInterface
{
    /**
     * @var int
     */
    const HTTP_CODE_CONFLICT = 409;

    /**
     * @var int
     */
    const HTTP_CODE_NOT_FOUND = 404;

    /**
     * @var int
     */
    const HTTP_CODE_OK = 200;

    /**
     * @var int
     */
    const HTTP_CODE_CREATED = 201;

    /**
     * @var int
     */
    const HTTP_CODE_NO_CONTENT = 204;

    /**
     * Returns an encryption flag
     *
     * @return boolean
     */
    public function isEncryptionInUse();

    /**
     * Sets encryption flag
     *
     * @param boolean $useEncryption
     * @return FifaConnectServiceBusClient
     */
    public function setUseEncryption($useEncryption);

    /**
     * Sends a message to a specified recipient via Message Bus.
     *
     * @param string $recipient
     * @param Message $message
     * @return null
     *
     * @throws InvalidClientDataException
     * @throws QueueNotFoundException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function send($recipient, Message $message);

    /**
     * Gets a message from the Message Bus. Message is deleted from the queue.
     *
     * @param int $timeout
     * @return Message
     * 
     * @throws NoMessagesAvailableException
     * @throws InvalidClientDataException
     * @throws QueueNotFoundException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function receive($timeout = 60);

    /**
     * Gets and locks a message without deleting it from the queue. If message is not deleted within configured time it's automatically unlocked and visible for other recipients.
     *
     * @param int $timeout
     * @return Message
     * 
     * @throws NoMessagesAvailableException
     * @throws QueueNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function peekLock($timeout = 60);

    /**
     * Deletes a previously locked message (see peekLock method).
     *
     * @param string $messageId
     * @param string $lockToken
     * @return null
     * @throws MessageNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function delete($messageId, $lockToken);

    /**
     * Unlocks a previously locked message. The message is returned to the queue and can be fetched using peekLock or receive method.
     *
     * @param string $messageId
     * @param string $lockToken
     * @return null
     * @throws MessageNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function unlock($messageId, $lockToken);

    /**
     * Extends the lock time for the previously locked message (see peekLock method).
     *
     * @param string $messageId
     * @param string $lockToken
     * @return null
     * 
     * @throws MessageNotFoundException
     * @throws InvalidClientDataException
     * @throws FifaConnectServiceBusSdkException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function renewLock($messageId, $lockToken);

    /**
     * Creates a topic with given name
     *
     * @param $topicName
     *
     * @throws FifaConnectServiceBusSdkException
     * @throws InvalidClientDataException
     * @throws MessageNotFoundException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function createTopic($topicName);

    /**
     * Deletes a topic with given name
     *
     * @param $topicName
     *
     * @throws FifaConnectServiceBusSdkException
     * @throws InvalidClientDataException
     * @throws MessageNotFoundException
     * @throws UnauthorizedException
     * @throws AuthenticationException
     */
    public function deleteTopic($topicName);
}