<?php
namespace Fifa\ConnectServiceBus\Sdk\Message;

/**
 * Class Message
 * @package Fifa\ConnectServiceBus\Sdk\Message
 */
class Message
{
    const LOCK_TOKEN_KEY = 'LockToken';
    const MESSAGE_ID_KEY = 'MessageId';

    const SENDER_KEY = 'Sender';
    const REPLY_TO_KEY = 'ReplyTo';

    /**
     * @var string
     */
    private $content;

    /**
     * @var string
     */
    private $action;

    /**
     * @param array
     */
    private $brokerProperties = array();

    /**
     * @var array
     */
    private $properties = array();

    /**
     * Constructor
     *
     * @param string $content
     * @param null|string $action
     * @param array $properties
     * @param array $brokerProperties
     */
    public function __construct($content, $action = null, array $properties = array(), array $brokerProperties = array())
    {
        $this->content = (string) $content;
        $this->action = is_null($action) ? null : (string) $action;
        $this->properties = $properties;
        $this->brokerProperties = $brokerProperties;
    }

    /**
     * String representation of the object
     *
     * @return string
     */
    public function __toString()
    {
        return "Action: {$this->action}, Content: {$this->content}, Properties: {$this->dumpProperties($this->properties)}, BrokerProperties: {$this->dumpProperties($this->brokerProperties)}";
    }

    /**
     * @param array $properties
     * @return string
     */
    private function dumpProperties(array $properties)
    {
        $propAsStrings = array();
        foreach ($properties as $key => $value) {
            $propAsStrings[] = $key . '=' . $value;
        }

        return join(', ', $propAsStrings);
    }

    /**
     * Gets message content
     *
     * @return string
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * Gets message action type
     *
     * @return string
     */
    public function getAction()
    {
        return $this->action;
    }

    /**
     * Gets message properties
     *
     * @return array
     */
    public function getProperties()
    {
        return $this->properties;
    }

    /**
     * Gets message broker properties
     *
     * @return array
     */
    public function getBrokerProperties()
    {
        return $this->brokerProperties;
    }

    /**
     * Gets message id
     *
     * @return string|null
     */
    public function getId()
    {
        return $this->getBrokerProperty(self::MESSAGE_ID_KEY);
    }

    /**
     * Gets lock token
     *
     * @return string|null
     */
    public function getLockToken()
    {
        return $this->getBrokerProperty(self::LOCK_TOKEN_KEY);
    }

    /**
     * Gets sender
     *
     * @return null|string
     */
    public function getSender()
    {
        return $this->getBrokerProperty(self::SENDER_KEY);
    }

    /**
     * Gets `reply to` value
     *
     * @return null|string
     */
    public function getReplyTo()
    {
        return $this->getBrokerProperty(self::REPLY_TO_KEY);
    }

    /**
     * Gets broker property, if exists
     *
     * @param string $key
     * @return null|string
     */
    private function getBrokerProperty($key)
    {
        return isset($this->brokerProperties[$key]) ? $this->brokerProperties[$key] : null;
    }

    /**
     * Sets the content of message
     *
     * @param string $content
     * @return Message
     */
    public function setContent($content)
    {
        $this->content = $content;
        return $this;
    }
}